'use strict';

/*
  Mijn Rooster – weekweergave (ma–zo)
  - Toont shifts per gebruiker per dag
  - Filters op afdelingen + "alleen ik"
  - Pauze-indicatie per shift:
      < 6 uur  => 0 minuten
      6..8 uur => 30 minuten
      > 8 uur  => 45 minuten
  - Als API break_minutes meestuurt, gebruiken we die; anders berekenen we het lokaal.
*/

(function(){
  // ======= DOM helpers
  const $ = (q)=>document.querySelector(q);
  const grid       = $('#rosterGrid');       // <table> of container voor rooster
  const checksWrap = $('#deptChecks');       // container voor afdeling-checkboxes
  const prevBtn    = $('#prevWeek');
  const nextBtn    = $('#nextWeek');
  const rangeLabel = $('#rangeLabel');
  const mineOnly   = $('#mineOnly');

  // ======= State
  let monday = getMonday(new Date());
  let myId = null;
  let selectedDepts = new Set();
  let deptPalette = new Map(); // deptId -> .c1..c8 kleurklassen

  // ======= Date helpers
  function getMonday(d){
    const x = new Date(d.getFullYear(), d.getMonth(), d.getDate());
    const day = x.getDay() || 7; // 1=ma ... 7=zo
    if (day !== 1) x.setDate(x.getDate() - (day - 1));
    x.setHours(0,0,0,0);
    return x;
  }
  function addDays(d,n){ return new Date(d.getFullYear(), d.getMonth(), d.getDate()+n); }
  function ymdLocal(d){
    const y = d.getFullYear(), m = String(d.getMonth()+1).padStart(2,'0'), dd = String(d.getDate()).padStart(2,'0');
    return `${y}-${m}-${dd}`;
  }
  function nlDate(d){ return d.toLocaleDateString('nl-NL',{weekday:'short',day:'2-digit',month:'2-digit'}); }

  // ======= Time helpers
  function clampToDay(startISO, endISO, dayStr){
    // knip een (mogelijk dag-overschrijdende) shift tot [dag 00:00, dag 23:59:59]
    const start = new Date((startISO||'').replace(' ','T'));
    const end   = new Date((endISO||'').replace(' ','T'));
    if (Number.isNaN(+start) || Number.isNaN(+end)) return null;
    const ds = new Date(dayStr+'T00:00:00');
    const de = new Date(dayStr+'T23:59:59');
    const a = new Date(Math.max(start, ds));
    const b = new Date(Math.min(end, de));
    if (a >= b) return null;
    return [a,b];
  }

  // ======= Pauze-regel
  function breakMinutesByDuration(mins){
    if (mins < 360) return 0;       // <6u
    if (mins <= 480) return 30;     // 6..8u
    return 45;                      // >8u
  }

  // ======= Data loaders
  async function loadMe(){
    try{
      const r = await fetch('/api/user/me.php'); const d = await r.json();
      if (d.ok){ myId = d.user.id; selectedDepts = new Set((d.user.department_ids||[]).map(Number)); }
    }catch(_){ /* noop */ }
  }

  async function loadDepartments(){
    try{
      const r = await fetch('/api/departments/list.php'); const d = await r.json();
      checksWrap.innerHTML = '';
      if (d.ok && Array.isArray(d.items) && d.items.length){
        // kies kleuren per afdeling
        deptPalette.clear();
        let i=1;
        d.items.forEach(dep=>{
          const id = `dept_${dep.id}`;
          const lab = document.createElement('label');
          lab.className = 'check';
          const checked = selectedDepts.size ? selectedDepts.has(Number(dep.id)) : i<=2; // preselecteer 2 als user niets heeft
          if (checked) selectedDepts.add(Number(dep.id));
          lab.innerHTML = `<input type="checkbox" value="${dep.id}" id="${id}" ${checked?'checked':''}><span>${escapeHtml(dep.name)}</span>`;
          const cb = lab.querySelector('input');
          cb.addEventListener('change', ()=>{
            if(cb.checked) selectedDepts.add(Number(cb.value)); else selectedDepts.delete(Number(cb.value));
            render(); // on change direct opnieuw laden
          });
          checksWrap.appendChild(lab);
          deptPalette.set(Number(dep.id), `c${((i-1)%8)+1}`); i++;
        });
      }
    }catch(_){ /* noop */ }
  }

  async function fetchData(){
    const start = ymdLocal(monday);
    const end   = ymdLocal(addDays(monday,6));
    rangeLabel.textContent = `${start} t/m ${end}`;

    const body = {
      start,
      end,
      department_ids: Array.from(selectedDepts)
    };
    const res = await fetch('/api/roster/list-range.php', {
      method:'POST',
      headers:{'Content-Type':'application/json'},
      body: JSON.stringify(body)
    });
    const data = await res.json();
    return data.ok ? data : { users:[], shifts:[] };
  }

  // ======= Render helpers
  function escapeHtml(s){ return (s||'').replace(/[&<>"]/g, c=>({ '&':'&amp;','<':'&lt;','>':'&gt;','"':'&quot;' }[c])); }

  function renderGrid(data){
    const dates = Array.from({length:7}, (_,i)=> ymdLocal(addDays(monday,i)));
    const users = (data.users||[]).filter(u => mineOnly.checked ? Number(u.id) === Number(myId) : true);

    // map[userId][dayStr] = [shifts...]
    const map = {};
    (data.shifts||[]).forEach(s=>{
      // API levert per assignment: user_id, plus shift velden
      if (mineOnly.checked && Number(s.user_id) !== Number(myId)) return;

      dates.forEach(day=>{
        const clipped = clampToDay(s.start_at, s.end_at, day);
        if (!clipped) return;

        const mins = Math.round((clipped[1] - clipped[0]) / 60000);
        // gebruik API break_minutes (van originele shift), anders bereken lokaal op basis van (geknipte) duur
        const pause = (typeof s.break_minutes === 'number')
          ? s.break_minutes
          : breakMinutesByDuration(mins);

        map[s.user_id] = map[s.user_id] || {};
        map[s.user_id][day] = map[s.user_id][day] || [];
        map[s.user_id][day].push({
          start: clipped[0].toTimeString().slice(0,5),
          end:   clipped[1].toTimeString().slice(0,5),
          depId: Number(s.department_id),
          depName: s.department_name,
          title: s.title || '',
          pause: pause
        });
      });
    });

    // ---- head
    let thead = `<thead><tr><th>Collega’s</th>${dates.map(d=>`<th>${nlDate(new Date(d+'T00:00:00'))}</th>`).join('')}</tr></thead>`;

    // ---- body
    let tbody = '<tbody>';
    users.forEach(u=>{
      tbody += `<tr><th class="usercell">${escapeHtml(u.name)}</th>`;
      dates.forEach(day=>{
        const items = (map[u.id] && map[u.id][day]) ? map[u.id][day] : [];
        if (!items.length){ tbody += '<td></td>'; return; }

        const chips = items.map(it=>{
          const cls = deptPalette.get(it.depId) || 'c1';
          const label = `${it.start}–${it.end}`;
          const hint = `${it.depName}${it.title? ' • '+it.title : ''} • Pauze ${it.pause} min`;
          return `<span class="badge ${cls}" title="${escapeHtml(hint)}"><span class="time">${escapeHtml(label)} • pauze ${it.pause}m</span></span>`;
        }).join('');

        tbody += `<td>${chips}</td>`;
      });
      tbody += '</tr>';
    });
    tbody += '</tbody>';

    grid.innerHTML = thead + tbody;
  }

  // ======= Main render
  async function render(){
    try{
      const data = await fetchData();
      renderGrid(data);
    }catch(e){
      grid.innerHTML = `<div class="error">Kon rooster niet laden.</div>`;
    }
  }

  // ======= Events
  prevBtn?.addEventListener('click', ()=>{ monday = addDays(monday,-7); render(); });
  nextBtn?.addEventListener('click', ()=>{ monday = addDays(monday, 7); render(); });
  mineOnly?.addEventListener('change', render);

  // ======= Init
  (async()=>{
    await loadMe();
    await loadDepartments();
    await render();
  })();
})();
